<?php declare(strict_types=1);
/**
 * This file is part of Swoft.
 *
 * @link     https://swoft.org
 * @document https://swoft.org/docs
 * @contact  group@swoft.org
 * @license  https://github.com/swoft-cloud/swoft/blob/master/LICENSE
 */

namespace MSwoft\WebSocket\Server\MessageParser;

use function preg_match;
use function strlen;
use MSwoft\Bean\Annotation\Mapping\Bean;
use MSwoft\WebSocket\Server\Contract\MessageParserInterface;
use MSwoft\WebSocket\Server\Message\Message;
use function explode;
use function strpos;
use function trim;

/**
 * Class TokenTextParser
 *
 * @since 2.0
 * @Bean()
 */
class TokenTextParser implements MessageParserInterface
{
    /**
     * Encode data to string.
     *
     * @param Message $message
     *
     * @return string
     */
    public function encode(Message $message): string
    {
        return $message->getDataString();
    }

    /**
     * Decode data to array
     *
     * @param string $data The raw message, use ':' to split cmd and data.
     * Format like:
     *  login:message body data
     * =>
     *  cmd: 'login'
     *  body: 'message body data'
     *
     * @return Message
     */
    public function decode(string $data): Message
    {
        $data = trim($data, ': ');

        // use default message command
        $cmd = '';
        if (strpos($data, ':') > 0) {
            [$cmd, $body] = explode(':', $data, 2);

            $cmd  = trim($cmd);
            $body = trim($body);

        // only an command
        } elseif (strlen($data) < 16 && 1 === preg_match('/^[\w-]+$/', $data)) {
            $cmd  = $data;
            $body = '';
        } else {
            $body = $data;
        }

        return Message::new($cmd, trim($body));
    }
}
