<?php declare(strict_types=1);

namespace MSwoft\Rpc\Client\Concern;

use MSwoft\Bean\BeanFactory;
use MSwoft\Connection\Pool\Exception\ConnectionPoolException;
use MSwoft\Log\Error;
use MSwoft\Rpc\Client\Connection;
use MSwoft\Rpc\Client\Exception\RpcClientException;
use MSwoft\Rpc\Client\Exception\RpcResponseException;
use MSwoft\Rpc\Client\Pool;
use MSwoft\Rpc\Client\ReferenceRegister;
use MSwoft\Rpc\Protocol;
use MSwoft\Stdlib\Helper\JsonHelper;

/**
 * Class ServiceTrait
 *
 * @since 2.0
 */
trait ServiceTrait
{
    /**
     * @param string $interfaceClass
     * @param string $methodName
     * @param array  $params
     *
     * @return mixed
     * @throws ConnectionPoolException
     * @throws RpcClientException
     * @throws RpcResponseException
     * @noinspection MagicMethodsValidityInspection
     */
    protected function __proxyCall(string $interfaceClass, string $methodName, array $params)
    {
        $poolName = ReferenceRegister::getPool(__CLASS__);
        $version  = ReferenceRegister::getVersion(__CLASS__);

        /* @var Pool $pool */
        $pool = BeanFactory::getBean($poolName);

        /* @var Connection $connection */
        $connection = $pool->getConnection();
        $connection->setRelease(true);
        $packet = $connection->getPacket();

        // Ext data
        $ext = $connection->getClient()->getExtender()->getExt();

        $protocol = Protocol::new($version, $interfaceClass, $methodName, $params, $ext);
        $data     = $packet->encode($protocol);
        $message  = sprintf('Rpc call failed.interface=%s method=%s pool=%s version=%s', $interfaceClass, $methodName,
            $poolName, $version);

        $result = $this->sendAndRecv($connection, $data, $message);
        $connection->release();

        $response = $packet->decodeResponse($result);
        if ($response->getError() !== null) {
            $code      = $response->getError()->getCode();
            $message   = $response->getError()->getMessage();
            $errorData = $response->getError()->getData();

            // Record rpc error message
            $errorMsg = sprintf('Rpc call error!code=%d message=%s data=%s pool=%s version=%s', $code, $message,
                JsonHelper::encode($errorData), $poolName, $version);

            Error::log($errorMsg);

            // Only to throw message and code
            throw new RpcResponseException($message, $code);
        }

        return $response->getResult();

    }

    /**
     * @param Connection $connection
     * @param string     $data
     * @param string     $message
     * @param bool       $reconnect
     *
     * @return string
     * @throws RpcClientException
     */
    private function sendAndRecv(Connection $connection, string $data, string $message, bool $reconnect = false): string
    {
        // Reconnect
        if ($reconnect) {
            $connection->reconnect();
        }

        if (!$connection->send($data)) {
            if ($reconnect) {
                throw new RpcClientException($message);
            }

            return $this->sendAndRecv($connection, $data, $message, true);
        }

        $result = $connection->recv();
        if ($result === false || empty($result)) {
            if ($reconnect) {
                throw new RpcClientException($message);
            }

            return $this->sendAndRecv($connection, $data, $message, true);
        }

        return $result;
    }
}
