<?php declare(strict_types=1);

namespace MSwoft\Log\Handler;

use DateTime;
use InvalidArgumentException;
use Monolog\Handler\AbstractProcessingHandler;
use MSwoft\Co;
use MSwoft\Exception\SwoftException;
use MSwoft\Log\Helper\Log;
use MSwoft\Log\Logger;
use MSwoft\Log\Logger as SwoftLogger;
use MSwoft\Stdlib\Helper\JsonHelper;
use Swlib\SaberGM;
use UnexpectedValueException;
use function alias;
use function array_column;
use function dirname;
use function implode;
use function in_array;
use function is_dir;

/**
 * Class FileHandler
 *
 * @since 2.0
 */
class WorkWxHandler extends AbstractProcessingHandler
{
    /**
     * Write log levels
     *
     * @var string
     */
    protected $levels = 'error,warning';

    /**
     * Write log file
     *
     * @var string
     */
    protected $notifyUrl = '';

    /**
     * @var array
     */
    protected $levelValues = [];

    /**
     * Will exec on construct
     */
    public function init(): void
    {
        if (is_array($this->levels)) {
            $this->levelValues = $this->levels;
            return;
        }

        // Levels like 'notice,error'
        if (is_string($this->levels)) {
            $levelNames        = explode(',', $this->levels);
            $this->levelValues = SwoftLogger::getLevelByNames($levelNames);
        }
    }

    /**
     * Write log by batch
     *
     * @param array $records
     *
     * @return void
     */
    public function handleBatch(array $records): void
    {
        $records = $this->recordFilter($records);
        if (!$records) {
            return;
        }

        $this->write($records);
    }

    /**
     * Write file
     *
     * @param array $records
     */
    protected function write(array $records): void
    {
        if (Log::getLogger()->isJson()) {
            $records = array_map([$this, 'formatJson'], $records);
        } else {
            $records = array_column($records, 'formatted');
        }
        $messageText = implode("\n", $records);

        if (Co::id() <= 0) {
            throw new InvalidArgumentException('Write log file must be under Coroutine!');
        }

        if (!class_exists("\\Swlib\\SaberGM")) {
            return;
        }

        if (empty($this->notifyUrl)) {
            return;
        }

        try {
            SaberGM::post($this->notifyUrl, null, [
                'json' => [
                    'msgtype'  => 'markdown',
                    "markdown" => [
                        'content' => trim($messageText)
                    ]
                ]
            ]);
        } catch (\Exception $exception) {
        }
    }

    /**
     * Filter record
     *
     * @param array $records
     *
     * @return array
     */
    private function recordFilter(array $records): array
    {
        $messages = [];
        foreach ($records as $record) {
            if (!isset($record['level'])) {
                continue;
            }
            if (!$this->isHandling($record)) {
                continue;
            }

            $record              = $this->processRecord($record);
            $record['formatted'] = $this->getFormatter()->format($record);

            $messages[] = $record;
        }
        return $messages;
    }

    /**
     * @param array $record
     *
     * @return string
     */
    public function formatJson(array $record): string
    {
        unset($record['formatted'], $record['extra']);
        if ($record['level'] === Logger::NOTICE) {
            unset($record['context']);
        }

        if ($record['datetime'] instanceof DateTime) {
            $record['datetime'] = $record['datetime']->format('Y-m-d H:i:s.u');
        }

        return JsonHelper::encode($record, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Whether to handler log
     *
     * @param array $record
     *
     * @return bool
     */
    public function isHandling(array $record): bool
    {
        if (empty($this->levelValues)) {
            return true;
        }

        return in_array($record['level'], $this->levelValues, true);
    }
}
