<?php declare(strict_types=1);

namespace MSwoft\Log\Handler;

use DateTime;
use Monolog\Handler\AbstractProcessingHandler;
use MSwoft\Console\Helper\Show;
use MSwoft\Log\Logger;
use MSwoft\Log\Logger as SwoftLogger;
use MSwoft\Stdlib\Helper\JsonHelper;
use function in_array;

/**
 * Class FileHandler
 *
 * @since 2.0
 */
class EchoHandler extends AbstractProcessingHandler
{
    /**
     * Write log levels
     *
     * @var string
     */
    protected $levels = '';

    /**
     * @var array
     */
    protected $levelValues = [];

    /**
     * Will exec on construct
     */
    public function init(): void
    {
        if (is_array($this->levels)) {
            $this->levelValues = $this->levels;
            return;
        }

        // Levels like 'notice,error'
        if (is_string($this->levels)) {
            $levelNames = explode(',', $this->levels);
            $this->levelValues = SwoftLogger::getLevelByNames($levelNames);
        }
    }

    /**
     * Write log by batch
     *
     * @param array $records
     *
     * @return void
     */
    public function handleBatch(array $records): void
    {
        $records = $this->recordFilter($records);
        if (!$records) {
            return;
        }

        $this->write($records);
    }

    /**
     * Write file
     *
     * @param array $records
     */
    protected function write(array $records): void
    {
        if (!env('APP_DEBUG')) {
            return;
        }

        foreach ($records as $record) {
            Show::block("[" . $record['datetime']->format('Y-m-d H:i:s.u') . "] " . $record['messages'], $record['level_name'], $record['level_name']);
        }
    }

    /**
     * Filter record
     *
     * @param array $records
     *
     * @return array
     */
    private function recordFilter(array $records): array
    {
        $messages = [];
        foreach ($records as $record) {
            if (!isset($record['level'])) {
                continue;
            }
            if (!$this->isHandling($record)) {
                continue;
            }

            $record = $this->processRecord($record);
            $record['formatted'] = $this->getFormatter()->format($record);

            $messages[] = $record;
        }
        return $messages;
    }

    /**
     * @param array $record
     *
     * @return string
     */
    public function formatJson(array $record): string
    {
        unset($record['formatted'], $record['extra']);
        if ($record['level'] === Logger::NOTICE) {
            unset($record['context']);
        }

        if ($record['datetime'] instanceof DateTime) {
            $record['datetime'] = $record['datetime']->format('Y-m-d H:i:s.u');
        }

        return JsonHelper::encode($record, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Whether to handler log
     *
     * @param array $record
     *
     * @return bool
     */
    public function isHandling(array $record): bool
    {
        if (empty($this->levelValues)) {
            return true;
        }

        return in_array($record['level'], $this->levelValues, true);
    }
}
