<?php declare(strict_types=1);

namespace MSwoft\Http\Server\Middleware;

use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;
use MSwoft\Bean\Annotation\Mapping\Bean;
use MSwoft\Bean\BeanFactory;
use MSwoft\Http\Message\Request;
use MSwoft\Http\Server\Contract\MiddlewareInterface;
use MSwoft\Http\Server\Router\Route;
use MSwoft\Http\Server\Router\Router;
use MSwoft\Validator\Exception\ValidatorException;
use MSwoft\Validator\ValidateRegister;
use MSwoft\Validator\Validator;
use function explode;

/**
 * Class ValidatorMiddleware
 *
 * @Bean()
 *
 * @since 2.0
 */
class ValidatorMiddleware implements MiddlewareInterface
{
    /**
     * @param ServerRequestInterface  $request
     * @param RequestHandlerInterface $handler
     *
     * @return ResponseInterface
     * @throws ValidatorException
     */
    public function process(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        /* @var Route $route */
        [$status, , $route] = $request->getAttribute(Request::ROUTER_ATTRIBUTE);
        if ($status !== Router::FOUND) {
            return $handler->handle($request);
        }

        // Controller and method
        $handlerId = $route->getHandler();
        [$className, $method] = explode('@', $handlerId);

        // Query validates
        $validates = ValidateRegister::getValidates($className, $method);
        if (empty($validates)) {
            return $handler->handle($request);
        }

        $data  = $request->getParsedBody();
        $query = $request->getQueryParams();
        $path = $route->getParams();

        // ParsedBody is empty string
        $parsedBody    = $data = empty($data) ? [] : $data;
        $notParsedBody = !is_array($data);
        if ($notParsedBody) {
            $parsedBody = [];
        }

        /* @var Validator $validator */
        $validator = BeanFactory::getBean('validator');

        /* @var Request $request */
        [$parsedBody, $query, $path] = $validator->validateRequest($parsedBody, $validates, $query, $path);
        if ($notParsedBody) {
            $parsedBody = $data;
        }

        $request = $request->withParsedBody($parsedBody)->withParsedQuery($query)->withParsedPath($path);

        return $handler->handle($request);
    }
}
