<?php declare(strict_types=1);

namespace MSwoft\Http\Server\Middleware;

use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;
use ReflectionException;
use ReflectionNamedType;
use Swoft;
use MSwoft\Bean\Annotation\Mapping\Bean;
use MSwoft\Bean\Container;
use MSwoft\Exception\SwoftException;
use MSwoft\Http\Message\Request;
use MSwoft\Http\Message\Response;
use MSwoft\Http\Server\Contract\MiddlewareInterface;
use MSwoft\Http\Server\Exception\MethodNotAllowedException;
use MSwoft\Http\Server\Exception\NotFoundRouteException;
use MSwoft\Http\Server\Router\Route;
use MSwoft\Http\Server\Router\Router;
use MSwoft\Stdlib\Helper\ObjectHelper;
use MSwoft\Stdlib\Helper\PhpHelper;
use function context;
use function explode;
use function sprintf;

/**
 * Class DefaultMiddleware
 *
 * @Bean()
 * @since 2.0
 */
class DefaultMiddleware implements MiddlewareInterface
{
    /**
     * @param ServerRequestInterface|Request $request
     * @param RequestHandlerInterface        $handler
     *
     * @return ResponseInterface
     * @throws MethodNotAllowedException
     * @throws NotFoundRouteException
     * @throws ReflectionException
     * @throws SwoftException
     */
    public function process(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        $method  = $request->getMethod();
        $uriPath = $request->getUriPath();

        /* @var Route $route */
        [$status, , $route] = $request->getAttribute(Request::ROUTER_ATTRIBUTE);

        // Not found
        if ($status === Router::NOT_FOUND) {
            throw new NotFoundRouteException("Route not found(path {$uriPath})!");
        }

        // Method not allowed
        if ($status === Router::METHOD_NOT_ALLOWED) {
            throw new MethodNotAllowedException(sprintf('Uri(%s) method(%s) not allowed!', $uriPath, $method));
        }

        // Controller and method
        $handlerId = $route->getHandler();
        [$className, $method] = explode('@', $handlerId);

        // Update context request
        context()->setRequest($request);

        $pathParams = $route->getParams();
        $bindParams = $this->bindParams($className, $method, $pathParams);
        $controller = Container::$instance->getSingleton($className);

        // Call class method
        $data = PhpHelper::call([$controller, $method], ...$bindParams);

        // Return is instanceof `ResponseInterface`
        if ($data instanceof ResponseInterface) {
            return $data;
        }

        $response = context()->getResponse();
        return $response->withData($data);
    }

    /**
     * Bind params
     *
     * @param string $className
     * @param string $method
     * @param array  $pathParams
     *
     * @return array
     * @throws ReflectionException
     * @throws SwoftException
     */
    private function bindParams(string $className, string $method, array $pathParams): array
    {
        $reflection   = Swoft::getReflection($className);
        $methodParams = $reflection['methods'][$method]['params'] ?? [];
        if (!$methodParams) {
            return [];
        }

        $bindParams = [];
        foreach ($methodParams as $methodParam) {
            [$paramName, $paramType, $paramDefaultType] = $methodParam;
            if (!$paramType instanceof ReflectionNamedType) {
                continue;
            }

            $type = $paramType->getName();
            if ($type === Request::class) {
                $bindParams[] = context()->getRequest();
            } elseif ($type === Response::class) {
                $bindParams[] = context()->getResponse();
            } elseif (isset($pathParams[$paramName])) {
                $bindParams[] = ObjectHelper::parseParamType($type, $pathParams[$paramName]);
            } else {
                $bindParams[] = $paramDefaultType ?? ObjectHelper::getDefaultValue($type);
            }
        }

        return $bindParams;
    }
}
