<?php declare(strict_types=1);

use MSwoft\Bean\BeanFactory;
use MSwoft\Config\Config;
use MSwoft\Context\Context;
use MSwoft\Contract\ContextInterface;
use MSwoft\Event\Manager\EventManager;
use MSwoft\Http\Server\HttpContext;
use MSwoft\Http\Server\HttpServer;
use MSwoft\Process\Context\ProcessContext;
use MSwoft\Process\Context\UserProcessContext;
use MSwoft\Rpc\Server\ServiceContext;
use MSwoft\Server\Context\ShutdownContext;
use MSwoft\Server\Context\StartContext;
use MSwoft\Server\Context\WorkerStartContext;
use MSwoft\Server\Context\WorkerStopContext;
use MSwoft\Server\Server;
use MSwoft\Task\FinishContext;
use MSwoft\Task\TaskContext;
use MSwoft\Validator\Exception\ValidatorException;
use MSwoft\Validator\Validator;
use MSwoft\WebSocket\Server\WebSocketServer;

if (!function_exists('env')) {
    /**
     * Gets the value of an environment variable.
     *
     * @param string $key
     * @param mixed  $default
     *
     * @return mixed
     */
    function env(string $key = null, $default = null)
    {
        if (!$key) {
            return $_SERVER;
        }

        $value = getenv($key);

        if ($value === false) {
            return value($default);
        }

        switch (strtolower($value)) {
            case 'on':
            case 'yes':
            case 'true':
            case '(true)':
                return true;
            case 'off':
            case 'no':
            case 'false':
            case '(false)':
                return false;
            case 'empty':
            case '(empty)':
                return '';
            case 'null':
            case '(null)':
                return null;
        }

        if (defined($value)) {
            $value = constant($value);
        }

        return $value;
    }
}

if (!function_exists('alias')) {
    /**
     * @param string $key
     *
     * @return string
     */
    function alias(string $key): string
    {
        return Swoft::getAlias($key);
    }
}

if (!function_exists('event')) {
    /**
     * @return EventManager
     */
    function event(): EventManager
    {
        return BeanFactory::getBean('eventManager');
    }
}

if (!function_exists('config')) {
    /**
     * Get value from config by key or default
     *
     * @param string $key
     * @param mixed  $default
     *
     * @return mixed
     */
    function config(string $key = null, $default = null)
    {
        if (!BeanFactory::hasBean('config')) {
            return sprintf('${.config.%s}', $key);
        }

        /* @var Config $config */
        $config = BeanFactory::getBean('config');

        return $config->get($key, $default);
    }
}

if (!function_exists('sgo')) {
    /**
     * Create coroutine like 'go()'
     * In the swoft, you must use `sgo()` instead of  swoole `go()` function
     *
     * @param callable $callable
     * @param bool     $wait
     *
     * @return int
     */
    function sgo(callable $callable, bool $wait = true): int
    {
        return \MSwoft\Co::create($callable, $wait);
    }
}

if (!function_exists('srun')) {
    /**
     * @param callable $callable
     *
     * @return bool
     */
    function srun(callable $callable): bool
    {
        return \MSwoft\Co::run($callable);
    }
}

if (!function_exists('context')) {
    /**
     * Get current context
     *
     * @return ContextInterface|HttpContext|ServiceContext|TaskContext|FinishContext|UserProcessContext|ProcessContext|StartContext|WorkerStartContext|WorkerStopContext|ShutdownContext
     */
    function context(): ContextInterface
    {
        return Context::get(true);
    }
}

if (!function_exists('server')) {
    /**
     * Get server instance
     *
     * @return Server|HttpServer|WebSocketServer
     */
    function server(): Server
    {
        return Server::getServer();
    }
}

if (!function_exists('validate')) {
    /**
     * @param array $data
     * @param string $validatorName
     * @param array $fields
     * @param array $userValidators
     * @param array $unfields
     *
     * @return array
     * @throws ValidatorException
     */
    function validate(
        array $data,
        string $validatorName,
        array $fields = [],
        array $userValidators = [],
        array $unfields = []
    ): array {
        /* @var Validator $validator */
        $validator = BeanFactory::getBean('validator');
        return $validator->validate($data, $validatorName, $fields, $userValidators, $unfields);
    }
}
