<?php declare(strict_types=1);

namespace MSwoft;

use Monolog\Formatter\LineFormatter;
use MSwoft\Config\Config;
use MSwoft\Helper\ComposerJSON;
use MSwoft\Log\Handler\EchoHandler;
use MSwoft\Log\Handler\FileHandler;
use MSwoft\Log\Logger;
use Throwable;
use function alias;
use function bean;
use function dirname;

/**
 * Class AutoLoader
 *
 * @since 2.0
 */
class AutoLoader extends SwoftComponent
{
    /**
     * Get namespace and dirs
     *
     * @return array
     */
    public function getPrefixDirs(): array
    {
        return [
            __NAMESPACE__ => __DIR__,
        ];
    }

    /**
     * Core bean definition
     *
     * @return array
     * @throws Throwable
     */
    public function beans(): array
    {
        return [
            'config' => [
                'class' => Config::class,
                'path' => alias('@config'),
                'parsers' => [
                    Config::TYPE_PHP => '${phpParser}'
                ]
            ],
            'lineFormatter' => [
                'class' => LineFormatter::class,
                'format' => '%datetime% [%level_name%] %messages%',
                'dateFormat' => 'H:i:s',
            ],
            'noticeHandler' => [
                'class' => FileHandler::class,
                'logFile' => '@runtime/logs/%d{Y-m-d}.log',  // 2.0.6 支持日志按时间切割
                'formatter' => bean('lineFormatter'),
                'levels' => 'error,info',
            ],
            'echoHandler' => [
                'class' => EchoHandler::class,
                'levels' => 'error,info,debug',
            ],
            'logger' => [
                'class' => Logger::class,
                'flushRequest' => false,
                'enable' => false,
                'handlers' => [
                    'notice' => bean('noticeHandler'),
                    'echo' => bean('echoHandler'),
                ],
            ]
        ];
    }

    /**
     * Metadata information for the component.
     *
     * @return array
     * @see ComponentInterface::getMetadata()
     */
    public function metadata(): array
    {
        $jsonFile = dirname(__DIR__) . '/composer.json';

        return ComposerJSON::open($jsonFile)->getMetadata();
    }
}
