<?php declare(strict_types=1);


namespace MSwoft\Cache\Aspect;

use MSwoft\Aop\Annotation\Mapping\Around;
use MSwoft\Aop\Annotation\Mapping\Aspect;
use MSwoft\Aop\Annotation\Mapping\PointAnnotation;
use MSwoft\Aop\Point\JoinPoint;
use MSwoft\Aop\Point\ProceedingJoinPoint;
use MSwoft\Cache\Annotaion\Mapping\CacheMethod;
use MSwoft\Cache\CacheHelper;
use MSwoft\Cache\CacheMethodRegister;
use MSwoft\Context\Context;
use MSwoft\Contract\ContextInterface;
use MSwoft\Http\Message\Request;

/**
 * @Aspect(order=1024)
 * @PointAnnotation(
 *     include={CacheMethod::class},
 *     exclude={}
 * )
 */
class CacheMethodAspect
{
    /**
     * 环绕通知
     *
     * @Around()
     *
     * @param ProceedingJoinPoint $proceedingJoinPoint
     * @param JoinPoint $joinPoint
     * @return mixed
     */
    public function aroundAdvice(ProceedingJoinPoint $proceedingJoinPoint, JoinPoint $joinPoint)
    {
        $annotation = CacheMethodRegister::getAnnotation($proceedingJoinPoint->getClassName(), $proceedingJoinPoint->getMethod());

        $ttl = intval($annotation->getTTL());
        $ctx = boolval($annotation->getCtx());
        $key = strval($annotation->getKey());
        $key = $this->getCacheKey($proceedingJoinPoint, $key);

        if ($ctx) {
            /** @var ContextInterface $context */
            $context = Context::get();
            if (is_null($context)) {
                return $proceedingJoinPoint->proceed();
            } else {
                $cache = $context->get($key);
                if (is_null($cache)) {
                    $cache = $proceedingJoinPoint->proceed();
                }
                $context->set($key, $cache);
                return $cache;
            }
        }

        return CacheHelper::get($key, $ttl, function () use ($key, $proceedingJoinPoint) {
            return $proceedingJoinPoint->proceed();
        });
    }

    private function getCacheKey(ProceedingJoinPoint $proceedingJoinPoint, $prefix = ''): string
    {
        $key  = '';
        $args = $proceedingJoinPoint->getArgsMap();
        // 如果设置过自定义缓存KEY
        if (!empty($prefix)) {
            // 没有参数的，直接返回
            if (empty($args)) {
                return $prefix;
            }

            if (count($args) === 1) {
                $val = $args[array_key_first($args)];
                return $prefix . ':' . (is_string($val) ? $val : floatval($val));
            }
            $key = $prefix;
        } else {
            $key = $proceedingJoinPoint->getClassName() . '@' . $proceedingJoinPoint->getMethod();
        }
        $params = [];
        foreach ($args as $k => $arg) {
            if ($arg instanceof Request) {
                $params[$k] = http_build_query($arg->input());
            } else if (is_string($arg) || is_numeric($arg)) {
                $params[$k] = $arg;
            } else if (is_bool($arg)) {
                $params[$k] = (float)$arg;
            } else if (is_array($arg)) {
                $params[$k] = http_build_query($arg);
            }
        }

        return $key . ":" . http_build_query($params);
    }
}