<?php declare(strict_types=1);


namespace MSwoft\Beanstalk;

use MSwoft\Bean\BeanFactory;
use MSwoft\Beanstalk\Connection\Connection;
use MSwoft\Beanstalk\Connection\ConnectionManager;
use MSwoft\Beanstalk\Exception\BeanstalkException;
use MSwoft\Connection\Pool\AbstractPool;
use MSwoft\Connection\Pool\Contract\ConnectionInterface;
use Pheanstalk\Contract\JobIdInterface;
use Pheanstalk\Contract\PheanstalkInterface;
use Pheanstalk\Contract\ResponseInterface;
use Pheanstalk\Job;
use Throwable;

/**
 * Class Pool
 *
 * @since 2.0
 *
 * @method array listTubes()
 * @method PheanstalkInterface useTube(string $tube)
 * @method PheanstalkInterface watch(string $tube)
 * @method PheanstalkInterface watchOnly(string $tube)
 * @method Job put(string $data, int $priority = PheanstalkInterface::DEFAULT_PRIORITY, int $delay = PheanstalkInterface::DEFAULT_DELAY, int $ttr = PheanstalkInterface::DEFAULT_TTR)
 * @method Job reserve()
 * @method Job reserveWithTimeout(int $timeout)
 * @method ResponseInterface stats()
 * @method ResponseInterface statsTube(string $tube)
 * @method void touch(JobIdInterface $job)
 * @method void delete(JobIdInterface $job)
 * @method void bury(JobIdInterface $job, int $priority = PheanstalkInterface::DEFAULT_PRIORITY)
 */
class Pool extends AbstractPool
{
    /**
     * Default pool
     */
    public const DEFAULT_POOL = 'beanstalk.pool';

    /**
     * @var BeanstalkDb
     */
    protected BeanstalkDb $beanstalkDb;

    /**
     * @return ConnectionInterface
     */
    public function createConnection(): ConnectionInterface
    {
        return $this->beanstalkDb->createConnection($this);
    }

    /**
     * call magic method
     *
     * @param string $name
     * @param array $arguments
     *
     * @return Connection
     * @throws BeanstalkException
     */
    public function __call(string $name, array $arguments)
    {
        try {
            /* @var ConnectionManager $conManager */
            $conManager = BeanFactory::getBean(ConnectionManager::class);

            $connection = $this->getConnection();

            $connection->setRelease(true);
            $conManager->setConnection($connection);
        } catch (Throwable $e) {
            throw new BeanstalkException(
                sprintf('Pool error is %s file=%s line=%d', $e->getMessage(), $e->getFile(), $e->getLine())
            );
        }

        // Not instanceof Connection
        if (!$connection instanceof Connection) {
            throw new BeanstalkException(
                sprintf('%s is not instanceof %s', get_class($connection), Connection::class)
            );
        }

        return $connection->{$name}(...$arguments);
    }

    /**
     * @return BeanstalkDb
     */
    public function getBeanstalkDb(): BeanstalkDb
    {
        return $this->beanstalkDb;
    }
}
