<?php declare(strict_types=1);

namespace MSwoft\Beanstalk\Connection;

use MSwoft\Bean\Annotation\Mapping\Bean;
use MSwoft\Bean\BeanFactory;
use MSwoft\Beanstalk\BeanstalkDb;
use MSwoft\Beanstalk\BeanstalkEvent;
use MSwoft\Beanstalk\Exception\BeanstalkException;
use MSwoft\Beanstalk\Overwrite\SocketFactory;
use MSwoft\Beanstalk\Pool;
use MSwoft\Connection\Pool\AbstractConnection;
use MSwoft\Log\Helper\Log;
use Pheanstalk\Contract\JobIdInterface;
use Pheanstalk\Contract\PheanstalkInterface;
use Pheanstalk\Contract\ResponseInterface;
use Pheanstalk\Job;
use Pheanstalk\Pheanstalk;
use Swoft;
use Throwable;

/**
 * Class Connection
 *
 * @since 2.0
 * @method array listTubes()
 * @method PheanstalkInterface useTube(string $tube)
 * @method PheanstalkInterface watch(string $tube)
 * @method PheanstalkInterface watchOnly(string $tube)
 * @method PheanstalkInterface ignore(string $tube)
 * @method Job put(string $data, int $priority = PheanstalkInterface::DEFAULT_PRIORITY, int $delay = PheanstalkInterface::DEFAULT_DELAY, int $ttr = PheanstalkInterface::DEFAULT_TTR)
 * @method Job reserve()
 * @method Job reserveWithTimeout(int $timeout)
 * @method ResponseInterface stats()
 * @method ResponseInterface statsTube(string $tube)
 * @method void touch(JobIdInterface $job)
 * @method void delete(JobIdInterface $job)
 * @method void bury(JobIdInterface $job, int $priority = PheanstalkInterface::DEFAULT_PRIORITY)
 *
 * @Bean(scope=Bean::PROTOTYPE)
 */
class Connection extends AbstractConnection
{
    /**
     * @var PheanstalkInterface
     */
    protected $client;

    /**
     * @var BeanstalkDb
     */
    protected $beanstalkDb;

    protected $supportedMethods = [
        'listTubes',
        'useTube',
        'watch',
        'watchOnly',
        'put',
        'reserve',
        'reserveWithTimeout',
        'stats',
        'statsTube',
        'touch',
        'delete',
        'bury',
        'ignore'
    ];

    protected $autoRelease = [
        'listTube',
        'statsTube',
        'stats',
        'put'
    ];

    /**
     * @param Pool $pool
     * @param BeanstalkDb $beanstalkDb
     */
    public function initialize(Pool $pool, BeanstalkDb $beanstalkDb)
    {
        $this->pool        = $pool;
        $this->beanstalkDb = $beanstalkDb;
        $this->lastTime    = time();

        $this->id = $this->pool->getConnectionId();
    }

    public function create(): void
    {
        $this->createTime = time();
        $this->createClient();
    }

    /**
     * Close connection
     */
    public function close(): void
    {
        Swoft::trigger(BeanstalkEvent::BEFORE_CLOSE, $this);
        $this->client = null;
    }

    public function createClient(): void
    {
        $socket       = new SocketFactory(
            $this->beanstalkDb->getHost(),
            $this->beanstalkDb->getPort(),
            $this->beanstalkDb->getConnectTimeout(),
            $this->beanstalkDb->getCommandTimeout()
        );
        $this->client = Pheanstalk::createWithFactory($socket);

        Swoft::trigger(BeanstalkEvent::AFTER_CONNECT, $this, $this->client);
    }

    /**
     * @param bool $force
     *
     */
    public function release(bool $force = false): void
    {
        /* @var ConnectionManager $conManager */
        $conManager = BeanFactory::getBean(ConnectionManager::class);
        $conManager->releaseConnection($this->id);

        parent::release($force);
    }

    /**
     * @return bool
     */
    public function reconnect(): bool
    {
        try {
            $this->client->reconnect();
        } catch (Throwable $e) {
            Log::error('Beanstalk reconnect error(%s)', $e->getMessage());
            return false;
        }

        return true;
    }

    public function __call(string $method, array $parameters)
    {
        return $this->command($method, $parameters);
    }

    public function command(string $method, array $parameters = [], bool $reconnect = false)
    {
        try {
            if (!in_array($method, $this->supportedMethods, true)) {
                throw new BeanstalkException(
                    sprintf('Method(%s) is not supported!', $method)
                );
            }

            // Before event
            Swoft::trigger(BeanstalkEvent::BEFORE_COMMAND, null, $method, $parameters);

            $result = $this->client->$method(...$parameters);

            // After event
            Swoft::trigger(BeanstalkEvent::AFTER_COMMAND, null, $method, $parameters, $result);

            if (in_array($method, $this->autoRelease, true)) {
                $this->release();
            }
        } catch (Throwable $throwable) {
            if (!$reconnect && $this->reconnect()) {
                return $this->command($method, $parameters, true);
            }
            $this->pool->remove();
            throw new BeanstalkException(
                sprintf('Beanstalk command reconnect error(%s)', $throwable->getMessage())
            );
        }

        return $result;
    }
}
