<?php declare(strict_types=1);


namespace MSwoft\Beanstalk;

use MSwoft\Bean\BeanFactory;
use MSwoft\Beanstalk\Connection\Connection;
use MSwoft\Beanstalk\Connection\ConnectionManager;
use MSwoft\Beanstalk\Exception\BeanstalkException;
use Pheanstalk\Contract\JobIdInterface;
use Pheanstalk\Contract\PheanstalkInterface;
use Pheanstalk\Contract\ResponseInterface;
use Pheanstalk\Job;
use Throwable;

/**
 * Class Redis
 *
 * @since 2.0
 *
 * @method array listTubes()
 * @method PheanstalkInterface useTube(string $tube)
 * @method PheanstalkInterface watch(string $tube)
 * @method PheanstalkInterface watchOnly(string $tube)
 * @method PheanstalkInterface ignore(string $tube)
 * @method Job put(string $data, int $priority = PheanstalkInterface::DEFAULT_PRIORITY, int $delay = PheanstalkInterface::DEFAULT_DELAY, int $ttr = PheanstalkInterface::DEFAULT_TTR)
 * @method Job reserve()
 * @method Job reserveWithTimeout(int $timeout)
 * @method ResponseInterface stats()
 * @method ResponseInterface statsTube(string $tube)
 * @method void touch(JobIdInterface $job)
 * @method void delete(JobIdInterface $job)
 * @method void bury(JobIdInterface $job, int $priority = PheanstalkInterface::DEFAULT_PRIORITY)
 */
class Beanstalk
{
    /**
     * @param string $pool
     *
     * @return Connection
     * @throws BeanstalkException
     */
    public static function connection(string $pool = Pool::DEFAULT_POOL): Connection
    {
        try {
            /* @var ConnectionManager $conManager */
            $conManager = BeanFactory::getBean(ConnectionManager::class);

            /* @var Pool $redisPool */
            $redisPool  = BeanFactory::getBean($pool);
            $connection = $redisPool->getConnection();

            $connection->setRelease(true);
            $conManager->setConnection($connection);
        } catch (Throwable $e) {
            throw new BeanstalkException(
                sprintf('Pool error is %s file=%s line=%d', $e->getMessage(), $e->getFile(), $e->getLine())
            );
        }

        // Not instanceof Connection
        if (!$connection instanceof Connection) {
            throw new BeanstalkException(
                sprintf('%s is not instanceof %s', get_class($connection), Connection::class)
            );
        }
        return $connection;
    }

    /**
     * @param string $method
     * @param array  $arguments
     *
     * @return mixed
     * @throws BeanstalkException
     */
    public static function __callStatic(string $method, array $arguments)
    {
        $connection = self::connection();
        return $connection->{$method}(...$arguments);
    }
}
